    // Copyright 2006 Nemanja Trifunovic
     
    /*
    Permission is hereby granted, free of charge, to any person or organization
    obtaining a copy of the software and accompanying documentation covered by
    this license (the "Software") to use, reproduce, display, distribute,
    execute, and transmit the Software, and to prepare derivative works of the
    Software, and to permit third-parties to whom the Software is furnished to
    do so, all subject to the following:
     
    The copyright notices in the Software and this entire statement, including
    the above license grant, this restriction and the following disclaimer,
    must be included in all copies of the Software, in whole or in part, and
    all derivative works of the Software, unless such copies or derivative
    works are solely in the form of machine-executable object code generated by
    a source language processor.
     
    THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
    IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
    FITNESS FOR A PARTICULAR PURPOSE, TITLE AND NON-INFRINGEMENT. IN NO EVENT
    SHALL THE COPYRIGHT HOLDERS OR ANYONE DISTRIBUTING THE SOFTWARE BE LIABLE
    FOR ANY DAMAGES OR OTHER LIABILITY, WHETHER IN CONTRACT, TORT OR OTHERWISE,
    ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
    DEALINGS IN THE SOFTWARE.
    */
     
    #ifndef UTF8_FOR_CPP_CHECKED_H_2675DCD0_9480_4c0c_B92A_CC14C027B731
    #define UTF8_FOR_CPP_CHECKED_H_2675DCD0_9480_4c0c_B92A_CC14C027B731
     
    #include <stdexcept>
    #include "core.h"
     
    #include "asf/core/Logger.h"
     
    DEFINE_LOGGER_AND_LEVEL(_logger, "/asf/codecs/utf8", Info);
     
    namespace utf8 {
    /// The library API - functions intended to be called by the users
     
    template < typename octet_iterator >
    octet_iterator append(uint32_t cp, octet_iterator result) {
        LOG_ASSERT_FATAL_MSG_STATIC(utf8::internal::is_code_point_valid(cp), "invalid_code_point");
     
        if (cp < 0x80)  // one octet
            *(result++) = static_cast< uint8_t >(cp);
        else if (cp < 0x800) {  // two octets
            *(result++) = static_cast< uint8_t >((cp >> 6) | 0xc0);
            *(result++) = static_cast< uint8_t >((cp & 0x3f) | 0x80);
        } else if (cp < 0x10000) {  // three octets
            *(result++) = static_cast< uint8_t >((cp >> 12) | 0xe0);
            *(result++) = static_cast< uint8_t >(((cp >> 6) & 0x3f) | 0x80);
            *(result++) = static_cast< uint8_t >((cp & 0x3f) | 0x80);
        } else {  // four octets
            *(result++) = static_cast< uint8_t >((cp >> 18) | 0xf0);
            *(result++) = static_cast< uint8_t >(((cp >> 12) & 0x3f) | 0x80);
            *(result++) = static_cast< uint8_t >(((cp >> 6) & 0x3f) | 0x80);
            *(result++) = static_cast< uint8_t >((cp & 0x3f) | 0x80);
        }
        return result;
    }
     
    template < typename octet_iterator, typename output_iterator >
    output_iterator replace_invalid(octet_iterator start,
                                    octet_iterator end,
                                    output_iterator out,
                                    uint32_t replacement) {
        while (start != end) {
            octet_iterator sequence_start = start;
            internal::utf_error err_code = utf8::internal::validate_next(start, end);
            switch (err_code) {
            case internal::UTF8_OK:
                for (octet_iterator it = sequence_start; it != start; ++it) *out++ = *it;
                break;
            case internal::NOT_ENOUGH_ROOM:
                LOG_FATAL_STATIC("not_enough_room");
            case internal::INVALID_LEAD:
                out = utf8::append(replacement, out);
                ++start;
                break;
            case internal::INCOMPLETE_SEQUENCE:
            case internal::OVERLONG_SEQUENCE:
            case internal::INVALID_CODE_POINT:
                out = utf8::append(replacement, out);
                ++start;
                // just one replacement mark for the sequence
                while (start != end && utf8::internal::is_trail(*start)) ++start;
                break;
            }
        }
        return out;
    }
     
    template < typename octet_iterator, typename output_iterator >
    inline output_iterator replace_invalid(octet_iterator start,
                                           octet_iterator end,
                                           output_iterator out) {
        static const uint32_t replacement_marker = utf8::internal::mask16(0xfffd);
        return utf8::replace_invalid(start, end, out, replacement_marker);
    }
     
    template < typename octet_iterator >
    uint32_t next(octet_iterator& it, octet_iterator end) {
        uint32_t cp = 0;
        internal::utf_error err_code = utf8::internal::validate_next(it, end, cp);
        switch (err_code) {
        case internal::UTF8_OK:
            break;
        case internal::NOT_ENOUGH_ROOM:
            LOG_FATAL_STATIC("not_enough_room");
        case internal::INVALID_LEAD:
        case internal::INCOMPLETE_SEQUENCE:
        case internal::OVERLONG_SEQUENCE:
            LOG_FATAL_STATIC("invalid_utf8");
        case internal::INVALID_CODE_POINT:
            LOG_FATAL_STATIC("invalid_code_point");
        }
        return cp;
    }
     
    template < typename octet_iterator >
    uint32_t peek_next(octet_iterator it, octet_iterator end) {
        return utf8::next(it, end);
    }
     
    template < typename octet_iterator >
    uint32_t prior(octet_iterator& it, octet_iterator start) {
        // can't do much if it == start
        if (it == start) {
            LOG_FATAL_STATIC("not_enough_room");
        }
     
        octet_iterator end = it;
        // Go back until we hit either a lead octet or start
        while (utf8::internal::is_trail(*(--it)))
            if (it == start) {
                LOG_FATAL_STATIC("invalid_utf8");  // error - no lead byte in the sequence
            }
        return utf8::peek_next(it, end);
    }
     
    /// Deprecated in versions that include "prior"
    template < typename octet_iterator >
    uint32_t previous(octet_iterator& it, octet_iterator pass_start) {
        octet_iterator end = it;
        while (utf8::internal::is_trail(*(--it)))
            if (it == pass_start) {
                LOG_FATAL_STATIC("invalid_utf8");  // error - no lead byte in the sequence
            }
        octet_iterator temp = it;
        return utf8::next(temp, end);
    }
     
    template < typename octet_iterator, typename distance_type >
    void advance(octet_iterator& it, distance_type n, octet_iterator end) {
        for (distance_type i = 0; i < n; ++i) utf8::next(it, end);
    }
     
    template < typename octet_iterator >
    typename std::iterator_traits< octet_iterator >::difference_type distance(octet_iterator first,
                                                                              octet_iterator last) {
        typename std::iterator_traits< octet_iterator >::difference_type dist;
        for (dist = 0; first < last; ++dist) utf8::next(first, last);
        return dist;
    }
     
    template < typename u16bit_iterator, typename octet_iterator >
    octet_iterator utf16to8(u16bit_iterator start, u16bit_iterator end, octet_iterator result) {
        while (start != end) {
            // We decided to suppress the Lint warning for the 'codecs' related code, as it was
            // borrowed.
            //lint -e{661}
            //lint -e{662}
            uint32_t cp = utf8::internal::mask16(*start++);
            // Take care of surrogate pairs first
            if (utf8::internal::is_lead_surrogate(cp)) {
                if (start != end) {
                    //lint -e{661}
                    //lint -e{662}
                    uint32_t trail_surrogate = utf8::internal::mask16(*start++);
                    if (utf8::internal::is_trail_surrogate(trail_surrogate))
                        cp = (cp << 10) + trail_surrogate + internal::SURROGATE_OFFSET;
                    else {
                        LOG_FATAL_STATIC("invalid_utf16");
                    }
                } else {
                    LOG_FATAL_STATIC("invalid_utf16");
                }
     
            }
            // Lone trail surrogate
            else if (utf8::internal::is_trail_surrogate(cp)) {
                LOG_FATAL_STATIC("invalid_utf16");
            }
            result = utf8::append(cp, result);
        }
        return result;
    }
     
    template < typename u16bit_iterator, typename octet_iterator >
    u16bit_iterator utf8to16(octet_iterator start, octet_iterator end, u16bit_iterator result) {
        while (start != end) {
            uint32_t cp = utf8::next(start, end);
            if (cp > 0xffff) {  // make a surrogate pair
                *result++ = static_cast< uint16_t >((cp >> 10) + internal::LEAD_OFFSET);
                *result++ = static_cast< uint16_t >((cp & 0x3ff) + internal::TRAIL_SURROGATE_MIN);
            } else
                *result++ = static_cast< uint16_t >(cp);
        }
        return result;
    }
     
    template < typename octet_iterator, typename u32bit_iterator >
    octet_iterator utf32to8(u32bit_iterator start, u32bit_iterator end, octet_iterator result) {
        while (start != end) result = utf8::append(*(start++), result);
     
        return result;
    }
     
    template < typename octet_iterator, typename u32bit_iterator >
    u32bit_iterator utf8to32(octet_iterator start, octet_iterator end, u32bit_iterator result) {
        while (start != end) (*result++) = utf8::next(start, end);
     
        return result;
    }
     
    // The iterator class
    template < typename octet_iterator >
    class iterator : public std::iterator< std::bidirectional_iterator_tag, uint32_t > {
        octet_iterator it;
        octet_iterator range_start;
        octet_iterator range_end;
     
    public:
        iterator() {}
        explicit iterator(const octet_iterator& octet_it,
                          const octet_iterator& range_start,
                          const octet_iterator& range_end)
            : it(octet_it), range_start(range_start), range_end(range_end) {
            if (it < range_start || it > range_end) {
                LOG_FATAL_STATIC("Invalid utf-8 iterator position");
            }
        }
        // the default "big three" are OK
        octet_iterator base() const { return it; }
        uint32_t operator*() const {
            octet_iterator temp = it;
            return utf8::next(temp, range_end);
        }
        bool operator==(const iterator& rhs) const {
            if (range_start != rhs.range_start || range_end != rhs.range_end) {
                LOG_FATAL_STATIC("Comparing utf-8 iterators defined with different ranges");
            }
            return (it == rhs.it);
        }
        bool operator!=(const iterator& rhs) const { return !(operator==(rhs)); }
        iterator& operator++() {
            utf8::next(it, range_end);
            return *this;
        }
        iterator operator++(int) {
            iterator temp = *this;
            utf8::next(it, range_end);
            return temp;
        }
        iterator& operator--() {
            utf8::prior(it, range_start);
            return *this;
        }
        iterator operator--(int) {
            iterator temp = *this;
            utf8::prior(it, range_start);
            return temp;
        }
    };  // class iterator
     
    }  // namespace utf8
     
    #endif  // header guard